
import os, json
import numpy as np
import matplotlib.pyplot as plt

SEED = 20251020
rng = np.random.default_rng(SEED)

N = 256
Theta_ladder = [2, 3, 5]
kappa_ladder = ["8x8", "16x16"]

epsilon = 1e-2
beta    = 0.10
bandwidth = 1

do_sampling = True
n_samples   = 10_000

out_dir = "/mnt/data"
os.makedirs(out_dir, exist_ok=True)

def build_ring_adjacency(n, bandwidth=1):
    A = np.zeros((n, n), dtype=float)
    for i in range(n):
        for d in range(1, bandwidth+1):
            A[i, (i-d) % n] = 1.0
            A[i, (i+d) % n] = 1.0
    return A

def ring_bandwidth(A):
    n = A.shape[0]
    idx = np.argwhere(A > 0.0)
    if idx.size == 0:
        return 0
    d = np.minimum(np.abs(idx[:,1] - idx[:,0]), n - np.abs(idx[:,1] - idx[:,0]))
    return int(np.max(d))

def modeA_overlap_D(theta):
    O = 1.0 + np.cos(theta)
    D = (O - O.min()) / (O.max() - O.min())
    mask = np.ones_like(D, dtype=bool)
    return D * mask.astype(float), mask

def modeB_overlap_D(theta):
    D = np.ones_like(theta)
    D = (D - D.min()) / (D.max() - D.min() + 1e-12)
    mask = np.ones_like(D, dtype=bool)
    return D * mask.astype(float), mask

def build_kernel(D, A, epsilon, beta):
    n = D.shape[0]
    return epsilon * np.ones((n, n), dtype=float) + beta * A + np.diag(D)

def pf_power_iteration(K, tol=1e-12, max_iter=20_000):
    n = K.shape[0]
    v = np.ones(n, dtype=float) / n
    lam = None
    for it in range(1, max_iter+1):
        w = K @ v
        s = np.sum(np.abs(w))
        if s == 0:
            raise RuntimeError("Power iteration encountered zero vector.")
        v_next = w / s
        res = float(np.linalg.norm(v_next - v, 1))
        v = v_next
        if res < tol:
            break
    p = v / np.sum(v)
    lam = float((v @ (K @ v)) / max(np.dot(v, v), 1e-18))
    strictly_positive = bool(np.all(v > 0))
    return v, lam, it, res, strictly_positive, p

def fit_sinusoid(theta, y):
    X = np.column_stack([np.ones_like(theta), np.cos(theta), np.sin(theta)])
    coef, *_ = np.linalg.lstsq(X, y, rcond=None)
    yhat = X @ coef
    rms = float(np.sqrt(np.mean((y - yhat)**2)))
    mean_y = float(np.mean(y))
    norm_rms = float(rms / max(mean_y, 1e-18))
    return coef, yhat, norm_rms

def flatness_rms(y):
    ybar = float(np.mean(y))
    rms = float(np.sqrt(np.mean((y - ybar)**2)))
    return ybar, float(rms / max(ybar, 1e-18))

def visibility(y):
    ymax = float(np.max(y))
    ymin = float(np.min(y))
    return float((ymax - ymin) / max(ymax + ymin, 1e-18))

def topk(arr, k=5):
    idx = np.argsort(arr)[::-1][:k]
    return [(int(i), float(arr[i])) for i in idx]

theta = np.linspace(0.0, 2.0*np.pi, N, endpoint=False)
A = build_ring_adjacency(N, bandwidth=bandwidth)
bw_measured = ring_bandwidth(A)
wrap_ok = (A[0, N-1] > 0) and (A[N-1, 0] > 0)

D_A, mask_A = modeA_overlap_D(theta)
K_A = build_kernel(D_A, A, epsilon=epsilon, beta=beta)
v_A, lam_A, it_A, res_A, pos_A, p_A = pf_power_iteration(K_A)
coef_A, pred_A, rms_A = fit_sinusoid(theta, p_A)
vis_A = visibility(p_A)

idx = rng.choice(N, size=n_samples, p=p_A)
hist_A = np.bincount(idx, minlength=N) / float(n_samples)
L2_diff_A = float(np.linalg.norm(hist_A - p_A))

D_B, mask_B = modeB_overlap_D(theta)
K_B = build_kernel(D_B, A, epsilon=epsilon, beta=beta)
v_B, lam_B, it_B, res_B, pos_B, p_B = pf_power_iteration(K_B)
mean_B, rms_B = flatness_rms(p_B)
idx = rng.choice(N, size=n_samples, p=p_B)
hist_B = np.bincount(idx, minlength=N) / float(n_samples)
L2_diff_B = float(np.linalg.norm(hist_B - p_B))

shift_bins = 17
D_A_rot = np.roll(D_A, shift_bins)
K_A_rot = build_kernel(D_A_rot, A, epsilon=epsilon, beta=beta)
_, _, _, _, _, p_A_rot = pf_power_iteration(K_A_rot)
p_A_expected = np.roll(p_A, shift_bins)
recenter_rms = float(np.sqrt(np.mean((p_A_rot - p_A_expected)**2)) / max(np.mean(p_A_expected), 1e-18))
recenter_passed = recenter_rms < 1e-6

import matplotlib.pyplot as plt

plt.figure()
plt.plot(np.arange(N), p_A, label="PF $p$ (Mode A)")
plt.plot(np.arange(N), pred_A, linestyle="--", label="Sinusoid fit")
plt.xlabel("bin")
plt.ylabel("probability")
plt.title("Two‑Slit: Mode A (interference on) — PF $p$ and sinusoid fit")
plt.legend()
figA_path = os.path.join(out_dir, "two_slit_modeA_p_passing.png")
plt.savefig(figA_path, dpi=150, bbox_inches="tight")
plt.close()

plt.figure()
plt.plot(np.arange(N), p_B, label="PF $p$ (Mode B)")
plt.axhline(mean_B, linestyle="--", label="Flat mean")
plt.xlabel("bin")
plt.ylabel("probability")
plt.title("Two‑Slit: Mode B (which‑way) — PF $p$ and flat baseline")
plt.legend()
figB_path = os.path.join(out_dir, "two_slit_modeB_p_passing.png")
plt.savefig(figB_path, dpi=150, bbox_inches="tight")
plt.close()

plt.figure()
plt.plot(np.arange(N), p_A, label="PF $p$")
plt.plot(np.arange(N), hist_A, linestyle="--", label="Sampled $\\hat p$ (10k)")
plt.xlabel("bin")
plt.ylabel("probability")
plt.title("Mode A: PF vs sampled histogram (passing run)")
plt.legend()
histA_path = os.path.join(out_dir, "two_slit_modeA_hist_passing.png")
plt.savefig(histA_path, dpi=150, bbox_inches="tight")
plt.close()

plt.figure()
plt.plot(np.arange(N), p_B, label="PF $p$")
plt.plot(np.arange(N), hist_B, linestyle="--", label="Sampled $\\hat p$ (10k)")
plt.xlabel("bin")
plt.ylabel("probability")
plt.title("Mode B: PF vs sampled histogram (passing run)")
plt.legend()
histB_path = os.path.join(out_dir, "two_slit_modeB_hist_passing.png")
plt.savefig(histB_path, dpi=150, bbox_inches="tight")
plt.close()

manifest = {
    "demo": "two-slit-v2",
    "seed": int(SEED),
    "N_bins": int(N),
    "gates": {
        "Theta": Theta_ladder,
        "kappa": kappa_ladder
    },
    "kernel": {
        "epsilon": float(epsilon),
        "beta": float(beta),
        "adjacency": f"ring, bandwidth={bandwidth}, wrap"
    },
    "overlap_method": "normalized 1+cos(Δφ) for Mode A; flat for Mode B",
    "modes": [
        {"name": "interference_on", "co_eligibility": True},
        {"name": "which_way", "co_eligibility": False}
    ],
    "artifacts": {
        "fig_modeA": os.path.basename(figA_path),
        "fig_modeB": os.path.basename(figB_path),
        "hist_modeA": os.path.basename(histA_path),
        "hist_modeB": os.path.basename(histB_path)
    }
}
manifest_path = os.path.join(out_dir, "manifest_two_slit_v2_passing.json")
with open(manifest_path, "w") as f:
    json.dump(manifest, f, indent=2)

summary = {
    "seed": int(SEED),
    "kernel_params": {"epsilon": float(epsilon), "beta": float(beta)},
    "adjacency_bandwidth": int(bw_measured),
    "wrap_around": bool(wrap_ok),
    "mode_A": {
        "pf_iters": int(it_A),
        "pf_residual_L1": float(res_A),
        "principal_eigenvalue_est": float(lam_A),
        "strictly_positive": bool(pos_A),
        "top5_p": topk(p_A, 5),
        "rms_fringe_rel": float(rms_A),
        "visibility": float(vis_A),
        "sampling_L2_error_10k": float(L2_diff_A),
        "figure": os.path.basename(figA_path)
    },
    "mode_B": {
        "pf_iters": int(it_B),
        "pf_residual_L1": float(res_B),
        "principal_eigenvalue_est": float(lam_B),
        "strictly_positive": bool(pos_B),
        "top5_p": topk(p_B, 5),
        "rms_flat_rel": float(rms_B),
        "sampling_L2_error_10k": float(L2_diff_B),
        "figure": os.path.basename(figB_path)
    },
    "no_skip_locality_audit": {
        "bandwidth": int(bw_measured),
        "wrap_ok": bool(wrap_ok),
        "result": "pass" if (bw_measured == bandwidth and wrap_ok) else "fail"
    },
    "recenter_invariance": {
        "shift_bins": int(shift_bins),
        "rms_rel": float(recenter_rms),
        "result": "pass" if recenter_passed else "fail"
    },
    "manifest_file": os.path.basename(manifest_path)
}

print(json.dumps(summary, indent=2))
